<?php
/**
 * Plugin Name: RS Cron Viewer
 * Plugin URI: https://tools.olvy.dev/cron-viewer
 * Description: Displays all scheduled WP-Cron tasks in the admin panel.
 * Version: 1.0.10
 * Author: Robert Silven
 * Author URI: https://github.com/rsilven
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: rs-cron-viewer
 * Domain Path: /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class RS_Cron_Viewer {

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_menu' ) );
    }

    public function add_menu() {
        add_menu_page(
            esc_html__( 'Cron Tasks', 'rs-cron-viewer' ),           // Page title
            esc_html__( 'Cron Viewer', 'rs-cron-viewer' ),          // Menu title
            'manage_options',       // Capability
            'rs-cron-viewer',       // Menu slug
            array( $this, 'render_page' ), // Callback
            'dashicons-clock',      // Icon
            100                     // Position
        );
    }

    public function render_page() {
        // Retrieve all cron events
        $crons = _get_cron_array();
        $schedules = wp_get_schedules();
        $date_format = get_option( 'date_format' ) . ' ' . get_option( 'time_format' );

        echo '<div class="wrap">';
        echo '<h1>' . esc_html__( 'WP-Cron Tasks', 'rs-cron-viewer' ) . '</h1>';
        
        if ( empty( $crons ) ) {
            echo '<div class="notice notice-warning"><p>' . esc_html__( 'No scheduled tasks found.', 'rs-cron-viewer' ) . '</p></div>';
            echo '</div>';
            return;
        }

        echo '<table class="widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th style="width: 20%;">' . esc_html__( 'Next Run', 'rs-cron-viewer' ) . '</th>';
        echo '<th style="width: 25%;">' . esc_html__( 'Hook', 'rs-cron-viewer' ) . '</th>';
        echo '<th>' . esc_html__( 'Arguments', 'rs-cron-viewer' ) . '</th>';
        echo '<th style="width: 15%;">' . esc_html__( 'Schedule', 'rs-cron-viewer' ) . '</th>';
        echo '</tr></thead>';
        echo '<tbody>';

        foreach ( $crons as $timestamp => $cronhooks ) {
            foreach ( $cronhooks as $hook => $keys ) {
                foreach ( $keys as $k => $v ) {
                    
                    $schedule_label = esc_html__( 'One-off', 'rs-cron-viewer' );
                    if ( isset( $v['schedule'] ) && $v['schedule'] ) {
                        $schedule_label = isset($schedules[$v['schedule']]) ? $schedules[$v['schedule']]['display'] : $v['schedule'];
                    }

                    $time_diff = human_time_diff( time(), $timestamp );
                    $is_past = $timestamp < time();
                    
                    // Adjust time to local WP time
                    $local_time = get_date_from_gmt( date( 'Y-m-d H:i:s', $timestamp ), $date_format );

                    echo '<tr>';
                    
                    // Time column
                    echo '<td>';
                    echo '<strong>' . esc_html( $local_time ) . '</strong>';
                    echo '<br>';
                    if ( $is_past ) {
                        printf(
                            '<span style="color: #d63638;">(%s)</span>',
                            sprintf( esc_html__( 'overdue by %s', 'rs-cron-viewer' ), esc_html( $time_diff ) )
                        );
                    } else {
                        printf(
                            '<span style="color: #00a32a;">(%s)</span>',
                            sprintf( esc_html__( 'in %s', 'rs-cron-viewer' ), esc_html( $time_diff ) )
                        );
                    }
                    echo '</td>';

                    // Hook column
                    echo '<td><strong>' . esc_html( $hook ) . '</strong></td>';

                    // Args column
                    echo '<td>';
                    if ( ! empty( $v['args'] ) ) {
                        echo '<pre style="margin: 0; white-space: pre-wrap;">' . esc_html( print_r( $v['args'], true ) ) . '</pre>';
                    } else {
                        echo '<span style="color: #aaa;">' . esc_html__( 'No arguments', 'rs-cron-viewer' ) . '</span>';
                    }
                    echo '</td>';

                    // Schedule column
                    echo '<td>' . esc_html( $schedule_label ) . '</td>';
                    echo '</tr>';
                }
            }
        }

        echo '</tbody>';
        echo '</table>';
        
        echo '<p style="margin-top: 20px; font-size: 13px; color: #666;">';
        printf(
            '%s <a href="https://tools.olvy.dev" target="_blank" style="text-decoration: none;">tools.olvy.dev</a> | ',
            esc_html__( 'More tools:', 'rs-cron-viewer' )
        );
        printf(
            '%s <a href="https://github.com/rsilven" target="_blank" style="text-decoration: none;">rsilven</a>',
            esc_html__( 'Author on GitHub:', 'rs-cron-viewer' )
        );
        echo '</p>';

        echo '</div>';
    }
}

new RS_Cron_Viewer();
